/* -------------------------------------------------------------------
   Tutorial: Optimization of Die Press Model (TEAM25)

   Features:
   - Nonlinear magnetostatic model of the die press
   - CAD parameters as ONELAB parameters
   - Direct approach of the shape sensitivity analysis
     based on the Lie derivative

   To compute the solution in a terminal:
       getdp shape.pro -solve GetPerformancesAndGradients

   To compute the solution interactively from the Gmsh GUI:
       File > Open > shape.pro
       Run (button at the bottom of the left panel)
   ------------------------------------------------------------------- */

/* This model computes the static magnetic field produced by a DC current. This
   corresponds to a "magnetostatic" physical model, obtained by combining the
   time-invariant Maxwell-Ampere equation (curl h = js, with h the magnetic
   field and js the source current density) with Gauss' law (Div b = 0, with b
   the magnetic flux density) and the magnetic constitutive law (b = mu h, with
   mu the magnetic permeability).

   Since Div b = 0, b can be derived from a vector magnetic potential a, such
   that b = curl a. Plugging this potential in Maxwell-Ampere's law and using
   the constitutive law leads to a vector Poisson equation in terms of the
   magnetic vector potential: curl(nu curl a) = js, where nu = 1/mu is
   the reluctivity.

   It is useful to share parameters with the user of the model,
   i.e., to make them editable in the GUI before running the model.
   Such variables are called ONELAB variables (because the sharing
   mechanism between the model and the GUI uses the ONELAB interface).
   ONELAB parameters are defined with a "DefineConstant" statement,
   which can be invoked in the .geo and .pro files.

   The optimization part is organized as follows:

   (1) We replaced both curves g-h and i-j by nurbs which have a certain
   of control points. We set the shape design variables as the distance
   of these control points from the origin point. Their actual value is
   provided by shape.py.

   (2) We make use of the solution of the nonlinear magnetostatic problem,
   for a given CAD configuration, to compute the objective function,
     w = Sum_{i=1}^{10} ((bpx_i-box_o)^2 + (bpy_i-boy_o)^2),
   as defined in TEAM25.

   (3) We compute the derivative of the flux density along the velocity
   field generated by the perturbation of each design variable, by means
   of the Lie derivative. The derivative of he objective, w, is therefore
   obtained by applying the chain rule. This is known as the direct approach
   of the sensitivity analysis, see [1].

   The value of the objective function as well as is derivative with respect
   to each design variable are provided to the ONELAB database. They can be
   accessed by shape.py.

   [1]:Erin Kuci, François Henrotte, Pierre Duysinx, and Christophe Geuzaine.
   ``Design sensitivity analysis for shape optimization based on the Lie derivative'',
   Computer Methods in Applied Mechanics and Engineering 317 (2017), pp. 702 –722.
   issn: 0045-7825.
*/

DefineConstant [
  Opt_ResDir = StrCat[CurrentDir,"res_opt/"]
  Opt_ResDir_Onelab = "Optimization/Results/"
  Model_smallAT = {1, Name "Model/Small Ampere-Turn", Choices{0,1}}
  Flag_PrintLevel = 1
  VelocityTag = -1
  // Nonlinear iterations
  Nb_max_iter = 30
  relaxation_factor = 1
  stop_criterion = 1e-5
  Optimizer = {0,Choices {0="Conveks - MMA",1="Conveks - GCMMA"},Name "Optimization/00Optimizer"}
  Opt_maxIter = {1600, Name "Optimization/01Max iterations"}
  Opt_maxChange = {0.0001, Name "Optimization/02Max change"}
  Opt_maxKKT = {0.001, Name "Optimization/03Max KKT norm"}
];

fobj = DefineNumber[0, Name "Optimization/Results/objective", 
	Graph "0300000000000000000000000000000000000"];
maxConstr = DefineNumber[0, Name "Optimization/Results/max(|Constraints|)", 
	Graph "0000000003000000000000000000000000000"];

Group {
  // One starts by giving explicit meaningful names to the Physical regions
  // defined in the "shape.geo" mesh file.
  Coil = Region[2];
  Core = Region[{1,3}];

  // Then one difines abstract regions so as to allow a generic definition of the
  // FunctionSpace, Formulation and PostProcessing fields with no reference to
  // model-specific Physical regions.
  Domain = Region[{Core,Coil,5}];
  NoFlux = Region[6];
}

Function{
  // Material law (here the magnetic reluctivity) is defined as piecewise
  // function (note the square brackets), in terms of the above defined
  // physical regions.

  mu0 = 4 * Pi * 1e-7;
  Mat1_b = {0.0,0.11,0.18,0.28,0.35,0.74,0.82,0.91,0.98,1.02,1.08,1.15,
    1.27,1.32,1.36,1.39,1.42,1.47,1.51,1.54,1.56,1.60,1.64,1.72};
  Mat1_h = {0.0,140.0,178.0,215.0,253.0,391.0,452.0,529.0,596.0,677.0,774.0,902.0,
    1164.0,1299.0,1462.0,1640.0,1851.0,2262.0,2685.0,3038.0,3395.0,4094.0,4756.0,7079.0};
  Mat1_b2 = List[Mat1_b]^2;
  Mat1_nu = List[Mat1_h]/List[Mat1_b];
  Mat1_nu(0) = Mat1_nu(1);
  Mat1_nu_b2  = ListAlt[Mat1_b2, Mat1_nu];
  nu[Core] = InterpolationLinear[ SquNorm[$1] ]{List[Mat1_nu_b2]} ;
  nu[Region[{Domain,-Core}]] = 1/mu0; // linear

  dnudb2[] = dInterpolationLinear[SquNorm[$1]]{List[Mat1_nu_b2]} ;
  dhdb_NL[Core] = 2*dnudb2[$1#1] * SquDyadicProduct[#1];

  // This is the current density which feeds the inductor.
  js[Coil] = Vector[0, 0, (Model_smallAT==1?4253:17500)/SurfaceArea[]{2}];

  // This is the desired x and y components of the induction field
  // in the cavity as defined in TEAM25.
  bx[] = (Model_smallAT==1?0.35:1.5) * X[]/Sqrt[X[]^2+Y[]^2];
  by[] = (Model_smallAT==1?0.35:1.5) * Y[]/Sqrt[X[]^2+Y[]^2];
}

Jacobian {
  { Name Vol; Case { { Region All ; Jacobian Vol; } } }
  { Name Sur; Case { { Region All ; Jacobian Sur; } } }
}

Integration{
  { Name Int2;
    Case{
      { Type Gauss;
        Case{
          { GeoElement Line;       NumberOfPoints 3; }
          { GeoElement Triangle;   NumberOfPoints 4; }
          { GeoElement Quadrangle; NumberOfPoints 4; }
        }
      }
    }
  }
}

// -------------------------------------------------------------------------
// The weak formulation for this problem is derived. The fields are
// vector-valued, and we have one linear (source) term in addition
// to the bilinear term.

Constraint{
  { Name Dirichlet;
    Case{
      { Region NoFlux; Type Assign; Value 0; }
    }
  }
}

FunctionSpace {
  { Name HCurl2D; Type Form1P;
    BasisFunction{
      { Name se1; NameOfCoef ae1; Function BF_PerpendicularEdge;
        Support Region[Domain]; Entity NodesOf[All]; }
    }
    Constraint{
      { NameOfCoef ae1; EntityType NodesOf; NameOfConstraint Dirichlet; }
    }
  }
}

Formulation{
  { Name MVP2D; Type FemEquation;
    Quantity{
      { Name a; Type Local; NameOfSpace HCurl2D; }
    }
    Equation{
      Galerkin{ [nu[{d a}] * Dof{d a}, {d a}];
        In Domain;  Jacobian Vol; Integration Int2; }
      Galerkin { JacNL [ dhdb_NL[{d a}] * Dof{d a} , {d a} ];
        In Core; Jacobian Vol ; Integration Int2 ; }
      Galerkin { [-js[], {a}];
        In Coil; Jacobian Vol; Integration Int2; }
    }
  }
}

// -------------------------------------------------------------------------
// Handling of the velocity field (through a fully-fixed function space
// defined on Domain).

Function{
  For i In {1:3}
    l_v~{i} = ListFromServer[Sprintf["Optimization/Results/velocity_%g_%g",VelocityTag,i]];
    velocity~{i}[] = ValueFromIndex[]{l_v~{i}()};
  EndFor
}

Constraint {
  For i In {1:3}
    { Name velocity~{i} ;
      Case {
        { Region Domain ; Value velocity~{i}[]; }
      }
    }
  EndFor
}

FunctionSpace {
  For i In {1:3}
    { Name H_v~{i} ; Type Form0;
      BasisFunction {
        { Name sn ; NameOfCoef un ; Function BF_Node ;
          Support Domain; Entity NodesOf[ All ] ; }
      }
      Constraint {
        { NameOfCoef un ; EntityType NodesOf ; NameOfConstraint velocity~{i}; }
      }
    }
  EndFor
}

// -------------------------------------------------------------------------
// Magnetic vector potential handling (through a fully-fixed function space
// defined on Domain).

Function{
  l_a = ListFromServer[StrCat[Opt_ResDir_Onelab,"a"]];
  aFromServer[] = ValueFromIndex[]{l_a()};
}

Constraint {
  { Name aFromServer;
    Case {
      { Region Domain ; Value aFromServer[]; }
      { Region NoFlux; Type Assign; Value 0; }
    }
  }
}

FunctionSpace{
  { Name HCurl2D_a_fullyfixed; Type Form1P;
    BasisFunction{
      { Name se1; NameOfCoef ae1; Function BF_PerpendicularEdge;
        Support Region[Domain]; Entity NodesOf[All]; }
    }
    Constraint{
      { NameOfCoef ae1; EntityType NodesOf; NameOfConstraint aFromServer; }
    }
  }
}

// -------------------------------------------------------------------------
// Handling of the direct sensitivity problem (depending on a given
// design variable represeted by the velocity field).

Function{
  dV[] = Transpose[Tensor[CompX[$1],CompX[$2],0,
            CompY[$1],CompY[$2],0,
            CompZ[$1],CompZ[$2],0]];

  // Lie derivative of H(B) where B is a 2-form and H a 1-form ($1:{d a}, $2:dV)
  LieOf_HB[] = nu[$1] * (Transpose[$2] * $1 - TTrace[$2] * $1 + $2 * $1) ;
  LieOf_HB_NL[] = dhdb_NL[$1] * (Transpose[$2] - TTrace[$2] * TensorDiag[1,1,1]) * $1;

  // Lie derivative of the 2-form Js ($1:Js[], $2:dV)
  LieOf_Js[] = TTrace[$2]*js[] - Transpose[$2]*js[];
}

FunctionSpace{
  { Name HCurl2D_Lva; Type Form1P ;
    BasisFunction {
      { Name se1 ; NameOfCoef ae1 ; Function BF_PerpendicularEdge ;
        Support Region[{ Domain}] ; Entity NodesOf [ All ] ; }
    }
    Constraint {
      { NameOfCoef ae1 ; EntityType NodesOf ; NameOfConstraint Dirichlet; }
    }
  }
}

Formulation {
  { Name DirectFormulationOf_MagSta_a_2D; Type FemEquation ;
    Quantity {
      { Name a; Type Local; NameOfSpace HCurl2D_a_fullyfixed; }
      { Name Lva; Type Local; NameOfSpace HCurl2D_Lva; }
      For i In {1:3}
        { Name v~{i} ; Type Local ; NameOfSpace H_v~{i}; }
      EndFor
    }
    Equation {
      Galerkin { [ nu[{d a}] * Dof{d Lva}, {d Lva} ];
        In Domain; Jacobian Vol; Integration Int2; }
      Galerkin { [ dhdb_NL[{d a}] * Dof{d Lva} , {d Lva} ];
        In Core; Jacobian Vol ; Integration Int2 ; }
      Galerkin { [LieOf_HB[{d a},dV[{d v_1},{d v_2},{d v_3}]],{d Lva}];
        In Domain; Jacobian Vol; Integration Int2; }
      Galerkin { [LieOf_HB_NL[{d a},dV[{d v_1},{d v_2},{d v_3}]],{d Lva}];
        In Core; Jacobian Vol; Integration Int2; }
      Galerkin { [ LieOf_Js[js[], dV[{d v_1},{d v_2},{d v_3}]], {Lva} ] ;
        In Coil ; Jacobian Vol; Integration Int2; }

      Galerkin { [ 0*Dof{a}, {a} ] ;
        In Domain; Jacobian Vol ; Integration Int2 ; }

      For i In {1:3}
        Galerkin { [ 0*Dof{v~{i}}, {v~{i}} ] ;
          In Domain; Jacobian Vol ; Integration Int2 ; }
      EndFor
    }
  }
}

// -------------------------------------------------------------------------
// In the following, we create the post-operations necessary
// for the computation of objective and constraints,
// as well as their sensitivities.

PostProcessing {
  { Name ObjectiveConstraints; NameOfFormulation MVP2D;
    Quantity {
      { Name w; Value{ Term{[(CompX[{d a}]-bx[])^2+(CompY[{d a}]-by[])^2 ]; In Domain; Jacobian Vol; } } }
      { Name az; Value{ Term{ [CompZ[{a}]]; In Domain; Jacobian Vol; } } }
      { Name b; Value{ Term{ [{d a}]; In Domain; Jacobian Vol; } } }
      { Name bTarget; Value{ Term{ [ Vector[bx[],by[],0]  ]; In Domain; Jacobian Vol; } } }
      { Name bMag; Value{ Term{ [Norm[{d a}]]; In Domain; Jacobian Vol; } } }
      { Name js; Value{ Term{ [js[]]; In Coil; Jacobian Vol; } } }
      { Name mur; Value{ Term{ [1/(nu[{d a}]*mu0)]; In Domain; Jacobian Vol; } } }
    }
  }
}

PostOperation {
  { Name Get_ObjectiveConstraints; NameOfPostProcessing ObjectiveConstraints;
    Operation{
      CreateDir[Opt_ResDir];
      Print[az, OnElementsOf Domain, Format NodeTable, File "",
        SendToServer StrCat[Opt_ResDir_Onelab,"a"], Hidden 1];
      Print[w,
        OnGrid {(9.5e-3+2.25e-3)*Cos[$A],(9.5e-3+2.25e-3)*Sin[$A],0}{0:50*Pi/180:5*Pi/180,0,0},
        Format SimpleTable, File StrCat[Opt_ResDir,"w.txt"]];
      Print[bMag,
        OnGrid {(9.5e-3+2.25e-3)*Cos[$A],(9.5e-3+2.25e-3)*Sin[$A],0}{0:50*Pi/180:5*Pi/180,0,0},
        Format SimpleTable, File StrCat[Opt_ResDir,"bMag.txt"]];
      Print[bTarget, OnGrid {(9.5e-3+2.25e-3)*Cos[$A],(9.5e-3+2.25e-3)*Sin[$A],0}{0:50*Pi/180:5*Pi/180,0,0},
        File StrCat[Opt_ResDir,"bTarget.pos"]];
      Print[b, OnGrid {(9.5e-3+2.25e-3)*Cos[$A],(9.5e-3+2.25e-3)*Sin[$A],0}{0:50*Pi/180:5*Pi/180,0,0},
        File StrCat[Opt_ResDir,"b.pos"]];
      If(Flag_PrintLevel>5)
        Print[mur, OnElementsOf Domain, File StrCat[Opt_ResDir,"mur.pos"]];
        Print[az, OnElementsOf Domain, File StrCat[Opt_ResDir,"az.pos"]];
        Print[a, OnElementsOf Domain, File StrCat[Opt_ResDir,"a.pos"]];
        Print[b, OnElementsOf Domain, File StrCat[Opt_ResDir,"b.pos"]];
        Print[bMag, OnElementsOf Domain, File StrCat[Opt_ResDir,"bMag.pos"]];
        Print[js, OnElementsOf Coil, File StrCat[Opt_ResDir,"js.pos"]];
      EndIf
    }
  }
}

// -------------------------------------------------------------------------
// Sensitivity of w based on a direct method

PostProcessing{
  { Name Direct_MagSta; NameOfFormulation DirectFormulationOf_MagSta_a_2D;
    PostQuantity {
      { Name Lie_w; Value { Term { [ 2*(CompX[{d a}]-bx[])*CompX[{d Lva}]
        +2*(CompY[{d a}]-by[])*CompY[{d Lva}] ] ; In Domain; Jacobian Vol; }}}
      { Name az; Value{ Term{ [CompZ[{a}]]; In Domain; Jacobian Vol; } } }
    }
  }
}

PostOperation{
  { Name Get_GradOf_w; NameOfPostProcessing Direct_MagSta;
    Operation {
      Print[Lie_w, OnGrid {(9.5e-3+2.25e-3)*Cos[$A],(9.5e-3+2.25e-3)*Sin[$A], 0} { 0:50*Pi/180:5*Pi/180, 0, 0 },
        Format SimpleTable, File StrCat[Opt_ResDir,Sprintf["Grad_w_wrt_dv_%g.txt",VelocityTag]]];
    }
  }
}

// Show useful data
PostProcessing {
  { Name Show_shape; NameOfFormulation DirectFormulationOf_MagSta_a_2D;
    PostQuantity {
      { Name VV; Value{Term{[ Vector[{v_1},{v_2},{v_3}] ];In Domain ; Jacobian Vol;}}}
      { Name Lie_az; Value { Term { [ CompZ[{Lva}] ] ; In Domain ; Jacobian Vol; }}}
    }
  }
}

PostOperation{
  { Name Show_shape; NameOfPostProcessing Show_shape;
    Operation {
      CreateDir[Opt_ResDir];
      Print[ VV, OnElementsOf Domain,File StrCat[Opt_ResDir, "velocity.pos"] ];
      Print[ Lie_az, OnElementsOf Domain,File StrCat[Opt_ResDir, "Lie_az.pos"] ];
    }
  }
}

// -------------------------------------------------------------------------
// The following resolution solves the physical problem
// to obtain the objective and the constraints.

Resolution {
  { Name GetPerformances;
    System {
      { Name SYS; NameOfFormulation MVP2D;}
    }
    Operation {
      // Initialization of the systems
      InitSolution[SYS];

      // Solve the nonlinear magnetostatic problem
      IterativeLoop[Nb_max_iter, stop_criterion, relaxation_factor]{
        GenerateJac[SYS];
        SolveJac[SYS];
        Evaluate[ SetNumberRunTime[$Iteration]{"iterNL"} ];
        Evaluate[ SetNumberRunTime[$Residual]{"residualNL"} ];
      }

      // Compute the objective function and the constraints
      PostOperation[Get_ObjectiveConstraints];
    }
  }
}

// -------------------------------------------------------------------------
// This resolution solves the direct problem to compute the sensitivity
// of the induction flux with respect to a given design variable.

Resolution {
  { Name GetGradient_wrt_dv;
    System {
      { Name DIR; NameOfFormulation DirectFormulationOf_MagSta_a_2D; }
    }
    Operation {
      InitSolution[DIR];
      Generate[DIR];
      Solve[DIR];
      PostOperation[Get_GradOf_w];
    }
  }
}
